﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/greengrassv2/GreengrassV2_EXPORTS.h>
#include <aws/greengrassv2/model/DeploymentStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GreengrassV2 {
namespace Model {

/**
 * <p>Contains information about a deployment.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/Deployment">AWS
 * API Reference</a></p>
 */
class Deployment {
 public:
  AWS_GREENGRASSV2_API Deployment() = default;
  AWS_GREENGRASSV2_API Deployment(Aws::Utils::Json::JsonView jsonValue);
  AWS_GREENGRASSV2_API Deployment& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GREENGRASSV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>
   * of the target IoT thing or thing group. When creating a subdeployment, the
   * targetARN can only be a thing group.</p>
   */
  inline const Aws::String& GetTargetArn() const { return m_targetArn; }
  inline bool TargetArnHasBeenSet() const { return m_targetArnHasBeenSet; }
  template <typename TargetArnT = Aws::String>
  void SetTargetArn(TargetArnT&& value) {
    m_targetArnHasBeenSet = true;
    m_targetArn = std::forward<TargetArnT>(value);
  }
  template <typename TargetArnT = Aws::String>
  Deployment& WithTargetArn(TargetArnT&& value) {
    SetTargetArn(std::forward<TargetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revision number of the deployment.</p>
   */
  inline const Aws::String& GetRevisionId() const { return m_revisionId; }
  inline bool RevisionIdHasBeenSet() const { return m_revisionIdHasBeenSet; }
  template <typename RevisionIdT = Aws::String>
  void SetRevisionId(RevisionIdT&& value) {
    m_revisionIdHasBeenSet = true;
    m_revisionId = std::forward<RevisionIdT>(value);
  }
  template <typename RevisionIdT = Aws::String>
  Deployment& WithRevisionId(RevisionIdT&& value) {
    SetRevisionId(std::forward<RevisionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the deployment.</p>
   */
  inline const Aws::String& GetDeploymentId() const { return m_deploymentId; }
  inline bool DeploymentIdHasBeenSet() const { return m_deploymentIdHasBeenSet; }
  template <typename DeploymentIdT = Aws::String>
  void SetDeploymentId(DeploymentIdT&& value) {
    m_deploymentIdHasBeenSet = true;
    m_deploymentId = std::forward<DeploymentIdT>(value);
  }
  template <typename DeploymentIdT = Aws::String>
  Deployment& WithDeploymentId(DeploymentIdT&& value) {
    SetDeploymentId(std::forward<DeploymentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the deployment.</p>
   */
  inline const Aws::String& GetDeploymentName() const { return m_deploymentName; }
  inline bool DeploymentNameHasBeenSet() const { return m_deploymentNameHasBeenSet; }
  template <typename DeploymentNameT = Aws::String>
  void SetDeploymentName(DeploymentNameT&& value) {
    m_deploymentNameHasBeenSet = true;
    m_deploymentName = std::forward<DeploymentNameT>(value);
  }
  template <typename DeploymentNameT = Aws::String>
  Deployment& WithDeploymentName(DeploymentNameT&& value) {
    SetDeploymentName(std::forward<DeploymentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the deployment was created, expressed in ISO 8601
   * format.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimestamp() const { return m_creationTimestamp; }
  inline bool CreationTimestampHasBeenSet() const { return m_creationTimestampHasBeenSet; }
  template <typename CreationTimestampT = Aws::Utils::DateTime>
  void SetCreationTimestamp(CreationTimestampT&& value) {
    m_creationTimestampHasBeenSet = true;
    m_creationTimestamp = std::forward<CreationTimestampT>(value);
  }
  template <typename CreationTimestampT = Aws::Utils::DateTime>
  Deployment& WithCreationTimestamp(CreationTimestampT&& value) {
    SetCreationTimestamp(std::forward<CreationTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the deployment.</p>
   */
  inline DeploymentStatus GetDeploymentStatus() const { return m_deploymentStatus; }
  inline bool DeploymentStatusHasBeenSet() const { return m_deploymentStatusHasBeenSet; }
  inline void SetDeploymentStatus(DeploymentStatus value) {
    m_deploymentStatusHasBeenSet = true;
    m_deploymentStatus = value;
  }
  inline Deployment& WithDeploymentStatus(DeploymentStatus value) {
    SetDeploymentStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether or not the deployment is the latest revision for its target.</p>
   */
  inline bool GetIsLatestForTarget() const { return m_isLatestForTarget; }
  inline bool IsLatestForTargetHasBeenSet() const { return m_isLatestForTargetHasBeenSet; }
  inline void SetIsLatestForTarget(bool value) {
    m_isLatestForTargetHasBeenSet = true;
    m_isLatestForTarget = value;
  }
  inline Deployment& WithIsLatestForTarget(bool value) {
    SetIsLatestForTarget(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parent deployment's target <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>
   * within a subdeployment.</p>
   */
  inline const Aws::String& GetParentTargetArn() const { return m_parentTargetArn; }
  inline bool ParentTargetArnHasBeenSet() const { return m_parentTargetArnHasBeenSet; }
  template <typename ParentTargetArnT = Aws::String>
  void SetParentTargetArn(ParentTargetArnT&& value) {
    m_parentTargetArnHasBeenSet = true;
    m_parentTargetArn = std::forward<ParentTargetArnT>(value);
  }
  template <typename ParentTargetArnT = Aws::String>
  Deployment& WithParentTargetArn(ParentTargetArnT&& value) {
    SetParentTargetArn(std::forward<ParentTargetArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_targetArn;
  bool m_targetArnHasBeenSet = false;

  Aws::String m_revisionId;
  bool m_revisionIdHasBeenSet = false;

  Aws::String m_deploymentId;
  bool m_deploymentIdHasBeenSet = false;

  Aws::String m_deploymentName;
  bool m_deploymentNameHasBeenSet = false;

  Aws::Utils::DateTime m_creationTimestamp{};
  bool m_creationTimestampHasBeenSet = false;

  DeploymentStatus m_deploymentStatus{DeploymentStatus::NOT_SET};
  bool m_deploymentStatusHasBeenSet = false;

  bool m_isLatestForTarget{false};
  bool m_isLatestForTargetHasBeenSet = false;

  Aws::String m_parentTargetArn;
  bool m_parentTargetArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace GreengrassV2
}  // namespace Aws
