// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F132

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_192_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xd4d9b1f0u, 0x9900a69cu, 0, 46 }, // d4d9b1f09900a69c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x49a930edu, 0x5fb47a45u, 0, 63 }, // 49a930ed5fb47a45 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x51d154c5u, 0x62096eb6u, 0, 80 }, // 51d154c562096eb6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x2cb77c05u, 0x3db2b2f6u, 0, 155 }, // 2cb77c053db2b2f6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x8d391fc7u, 0x5d384c42u, 40, 46 }, // 8d391fc75d384c42 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x1fbff4eau, 0x2d72be55u, 40, 63 }, // 1fbff4ea2d72be55 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xb4968481u, 0xcfaad203u, 40, 80 }, // b4968481cfaad203 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x59b0afdfu, 0x55a14469u, 97, 46 }, // 59b0afdf55a14469 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x4c162125u, 0x4ba32a0du, 97, 138 }, // 4c1621254ba32a0d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xbf15a046u, 0x6e6ac60au, 120, 80 }, // bf15a0466e6ac60a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xd8cf2d09u, 0x953b22e6u, 120, 155 }, // d8cf2d09953b22e6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x6705321cu, 0x7fbb92dau, 126, 46 }, // 6705321c7fbb92da = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x7ae4b0b3u, 0xd9dbc1a8u, 132, 138 }, // 7ae4b0b3d9dbc1a8 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_192_0_T_F_0___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 6, 8,12,12,12,12,12,12,12,12},
 { 3, 2, 3, 1, 1, 1, 7, 7, 7, 7},
 { 2, 2, 3, 1, 7, 7, 7, 7, 7, 7},
 { 3, 2, 1, 1, 7, 7, 7, 7, 7, 7},
 { 5, 5, 5, 4, 4,11,11,11,11,11},
 { 0, 0, 4, 9,10,10,10,10,10,10},
 { 5, 5, 4, 4,11,11,11,11,11, 7},
 { 5, 5, 5, 5,11,11,11,11,11,11},
 { 5, 5, 5, 5, 4,11,11,10,10,10},
 { 5, 5, 5, 4,11,11,11,11,11,11}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F132 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 192
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 192 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

