/* google_fit.rs
 *
 * Copyright 2020-2021 Rasmus Thomsen <oss@cogitri.dev>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

use super::{sync_provider::SyncProvider, DatabaseValue};
use crate::{
    core::{i18n_f, Settings},
    model::{Steps, Weight},
    prelude::*,
};
use anyhow::Result;
use gtk::glib;
use oauth2::{
    basic::{BasicClient, BasicTokenType},
    AuthUrl, ClientId, ClientSecret, CsrfToken, EmptyExtraTokenFields, PkceCodeChallenge,
    RedirectUrl, Scope, StandardTokenResponse, TokenResponse, TokenUrl,
};
use std::collections::HashMap;
use std::error::Error;
use uom::si::{f32::Mass, mass::kilogram};

static GOOGLE_PROVIDER_NAME: &str = "google-fit";
static GOOGLE_API_ENDPOINT: &str = "https://www.googleapis.com/fitness/v1";
static GOOGLE_AUTH_ENDPOINT_URL: &str = "https://accounts.google.com/o/oauth2/v2/auth";
static GOOGLE_TOKEN_ENDPOINT_URL: &str = "https://www.googleapis.com/oauth2/v3/token";
static GOOGLE_CLIENT_SECRET: &str = "QXYmZ982XWCdwKTW8mI3BbPp";
static GOOGLE_CLIENT_ID: &str =
    "652904425115-cdqjiporv9klugv9m7m0tpu44jt6cacb.apps.googleusercontent.com";

#[derive(serde::Deserialize)]
struct Value {
    #[serde(rename = "intVal")]
    pub int_val: Option<u32>,
    #[serde(rename = "fpVal")]
    pub fp_val: Option<f32>,
}

#[derive(serde::Deserialize)]
struct Point {
    #[serde(deserialize_with = "super::serialize::deserialize_modified_time_millis")]
    #[serde(rename = "modifiedTimeMillis")]
    pub date: glib::DateTime,
    pub value: Vec<Value>,
}

#[derive(serde::Deserialize)]
struct Points {
    pub point: Vec<Point>,
}

#[derive(Debug, Clone)]
pub struct GoogleFitSyncProvider {
    sender: glib::Sender<DatabaseValue>,
    token: Option<StandardTokenResponse<EmptyExtraTokenFields, BasicTokenType>>,
}

impl GoogleFitSyncProvider {
    /// Fetch steps from Google Fit.
    ///
    /// # Arguments
    /// * `date_opt` - If set, get steps from that date until now. Otherwise get all steps (e.g. for initial import).
    ///
    /// # Returns
    /// An array of [Steps], or a [anyhow::Error] if querying the Google Fit API fails.
    fn steps(&mut self, date_opt: Option<glib::DateTime>) -> Result<Vec<Steps>> {
        let points = if let Some(date) = date_opt {
            self.get::<Points>(&format!("users/me/dataSources/derived:com.google.step_count.delta:com.google.android.gms:merge_step_deltas/datasets/{}-{}", date.nanoseconds(), glib::DateTime::utc().nanoseconds()))
        } else {
            self.get::<Points>(&format!("users/me/dataSources/derived:com.google.step_count.delta:com.google.android.gms:merge_step_deltas/datasets/0-{}", glib::DateTime::utc().nanoseconds()))
        }?;

        Ok(Self::points_to_steps(points))
    }

    /// Fetch weight measurement from Google Fit.
    ///
    /// # Arguments
    /// * `date_opt` - If set, get weight measurements from that date until now. Otherwise get all weight measurements (e.g. for initial import).
    ///
    /// # Returns
    /// An array of [Weight]s, or a [anyhow::Error] if querying the Google Fit API fails.
    fn weights(&mut self, date_opt: Option<glib::DateTime>) -> Result<Vec<Weight>> {
        let points = if let Some(date) = date_opt {
            self.get::<Points>(&format!("users/me/dataSources/derived:com.google.weight:com.google.android.gms:merge_weight/datasets/{}-{}", date.nanoseconds(), glib::DateTime::utc().nanoseconds()))
        } else {
            self.get::<Points>(&format!("users/me/dataSources/derived:com.google.weight:com.google.android.gms:merge_weight/datasets/0-{}", glib::DateTime::utc().nanoseconds()))
        }?;

        Ok(Self::points_to_weights(points))
    }

    /// Convert Google-Fit JSON [Points] to an array of [Steps]
    // False-Positive in contains_key, we can't use .entry().insert_or() since we need the else condition
    #[allow(clippy::map_entry)]
    fn points_to_steps(p: Points) -> Vec<Steps> {
        let mut steps_map = HashMap::new();

        for s in p.point {
            if steps_map.contains_key(&s.date) {
                steps_map.insert(
                    s.date.clone(),
                    steps_map.get(&s.date).unwrap()
                        + s.value.iter().map(|s| s.int_val.unwrap()).sum::<u32>(),
                );
            } else {
                steps_map.insert(s.date, s.value.iter().map(|s| s.int_val.unwrap()).sum());
            }
        }

        steps_map
            .into_iter()
            .map(|(date, value)| Steps::new(date, value))
            .collect()
    }

    /// Convert Google-Fit JSON [Points] to an array of [Weight]
    fn points_to_weights(p: Points) -> Vec<Weight> {
        p.point
            .into_iter()
            .map(|p| {
                Weight::new(
                    p.date,
                    Mass::new::<kilogram>(p.value.last().unwrap().fp_val.unwrap_or(0.0)),
                )
            })
            .collect()
    }
}

impl SyncProvider for GoogleFitSyncProvider {
    fn provider_name(&self) -> &'static str {
        GOOGLE_PROVIDER_NAME
    }

    fn api_url(&self) -> &'static str {
        GOOGLE_API_ENDPOINT
    }

    fn oauth2_token(
        &mut self,
    ) -> Result<StandardTokenResponse<EmptyExtraTokenFields, BasicTokenType>> {
        if let Some(token) = &self.token {
            Ok(token.clone())
        } else {
            self.reauthenticate()?;
            Ok(self.token.clone().unwrap())
        }
    }

    /// Exchange the refresh token we already stored for an access token (which is valid for an hour).
    fn reauthenticate(&mut self) -> Result<()> {
        let client = BasicClient::new(
            ClientId::new(GOOGLE_CLIENT_ID.to_string()),
            Some(ClientSecret::new(GOOGLE_CLIENT_SECRET.to_string())),
            AuthUrl::new(GOOGLE_AUTH_ENDPOINT_URL.to_string()).unwrap(),
            Some(TokenUrl::new(GOOGLE_TOKEN_ENDPOINT_URL.to_string()).unwrap()),
        );
        self.token = Some(self.exchange_refresh_token(&client)?);
        Ok(())
    }

    /// Start the first authentication with Google Fit. This will open the user's
    /// browser so they can authenticate with Google and store the refresh token
    /// to the secret store.
    fn initial_authenticate(&mut self) -> Result<()> {
        let client = BasicClient::new(
            ClientId::new(GOOGLE_CLIENT_ID.to_string()),
            Some(ClientSecret::new(GOOGLE_CLIENT_SECRET.to_string())),
            AuthUrl::new(GOOGLE_AUTH_ENDPOINT_URL.to_string()).unwrap(),
            Some(TokenUrl::new(GOOGLE_TOKEN_ENDPOINT_URL.to_string()).unwrap()),
        )
        .set_redirect_uri(RedirectUrl::new("http://localhost:34981".to_string()).unwrap());
        let (pkce_code_challenge, pkce_code_verifier) = PkceCodeChallenge::new_random_sha256();
        let (authorize_url, csrf_state) = client
            .authorize_url(CsrfToken::new_random)
            .add_scope(Scope::new(
                "https://www.googleapis.com/auth/fitness.activity.read".to_string(),
            ))
            .add_scope(Scope::new(
                "https://www.googleapis.com/auth/fitness.activity.write".to_string(),
            ))
            .add_scope(Scope::new(
                "https://www.googleapis.com/auth/fitness.body.read".to_string(),
            ))
            .add_scope(Scope::new(
                "https://www.googleapis.com/auth/fitness.body.write".to_string(),
            ))
            .set_pkce_challenge(pkce_code_challenge)
            .url();

        let (code, returned_crfst_state) = Self::start_listen_server(authorize_url.as_str())?;

        if csrf_state.secret() != returned_crfst_state.secret() {
            anyhow::bail!(
                "{}",
                i18n_f(
                    "CRSF Verification failed, got {}, expected {}",
                    &[returned_crfst_state.secret(), csrf_state.secret()],
                )
            );
        }

        // Exchange the code with a token.
        self.token = Some(
            client
                .exchange_code(code)
                .set_pkce_verifier(pkce_code_verifier)
                .request(oauth2::ureq::http_client)
                .map_err(|e| {
                    anyhow::anyhow!(
                        "{}",
                        i18n_f(
                            "Requesting OAuth2 token failed due to error {}",
                            &[&e.source()
                                .map_or(String::new(), std::string::ToString::to_string)],
                        )
                    )
                })?,
        );

        if let Some(refresh_token) = self.token.as_ref().unwrap().refresh_token() {
            self.set_token(refresh_token.clone())?;
            let settings = Settings::instance();
            settings.set_sync_provider_setup_google_fit(true);
            settings.set_timestamp_last_sync_google_fit(glib::DateTime::local());
        }

        Ok(())
    }

    /// Start the initial import with Google Fit. This will import all data
    /// from Google Fit to the Tracker DB.
    fn initial_import(&mut self) -> Result<()> {
        let steps = self.steps(None)?;
        self.sender.send(DatabaseValue::Steps(steps)).unwrap();

        let weights = self.weights(None)?;
        self.sender.send(DatabaseValue::Weights(weights)).unwrap();

        Ok(())
    }

    /// Start the sync with Google Fit. This will sync data that has been added
    /// since the last sync.
    fn sync_data(&mut self) -> Result<()> {
        let settings = Settings::instance();
        let last_sync_date = settings.timestamp_last_sync_google_fit();
        settings.set_timestamp_last_sync_google_fit(glib::DateTime::local());

        let steps = self.steps(Some(last_sync_date.clone()))?;
        self.sender.send(DatabaseValue::Steps(steps)).unwrap();

        let weights = self.weights(Some(last_sync_date))?;
        self.sender.send(DatabaseValue::Weights(weights)).unwrap();

        Ok(())
    }
}

impl GoogleFitSyncProvider {
    pub fn new(sender: glib::Sender<DatabaseValue>) -> Self {
        Self {
            sender,
            token: None,
        }
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_steps_decode() {
        static STEPS_JSON: &str = "{\"minStartTimeNs\":\"1610233200000000000\",\"maxEndTimeNs\":\"1610750250193000000\",\"dataSourceId\":\"derived:com.google.step_count.delta:com.google.android.gms:merge_step_deltas\",\"point\":[{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610232600000000000\",\"endTimeNanos\":\"1610234640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610234640000000000\",\"endTimeNanos\":\"1610235180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610235180000000000\",\"endTimeNanos\":\"1610236140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610236140000000000\",\"endTimeNanos\":\"1610237040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610237040000000000\",\"endTimeNanos\":\"1610243280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610243280000000000\",\"endTimeNanos\":\"1610243460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610243460000000000\",\"endTimeNanos\":\"1610244660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610244660000000000\",\"endTimeNanos\":\"1610245380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610245380000000000\",\"endTimeNanos\":\"1610246580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610246580000000000\",\"endTimeNanos\":\"1610248080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610248080000000000\",\"endTimeNanos\":\"1610248320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610248320000000000\",\"endTimeNanos\":\"1610249100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610249100000000000\",\"endTimeNanos\":\"1610250420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610250420000000000\",\"endTimeNanos\":\"1610252040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610252040000000000\",\"endTimeNanos\":\"1610252340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610252340000000000\",\"endTimeNanos\":\"1610253060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610253060000000000\",\"endTimeNanos\":\"1610253420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610253420000000000\",\"endTimeNanos\":\"1610254920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610254920000000000\",\"endTimeNanos\":\"1610257860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610257860000000000\",\"endTimeNanos\":\"1610259420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610259420000000000\",\"endTimeNanos\":\"1610260440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610260440000000000\",\"endTimeNanos\":\"1610262540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610262540000000000\",\"endTimeNanos\":\"1610263800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610263800000000000\",\"endTimeNanos\":\"1610264640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610264640000000000\",\"endTimeNanos\":\"1610265540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610265540000000000\",\"endTimeNanos\":\"1610266140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610266140000000000\",\"endTimeNanos\":\"1610266620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610266860000000000\",\"endTimeNanos\":\"1610266920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":29}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610266920000000000\",\"endTimeNanos\":\"1610266981000000000\",\"value\":[{\"mapVal\":[],\"intVal\":26}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610266981000000000\",\"endTimeNanos\":\"1610267040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610267340000000000\",\"endTimeNanos\":\"1610267400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610268420000000000\",\"endTimeNanos\":\"1610268480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610268540000000000\",\"endTimeNanos\":\"1610268600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":45}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610269740000000000\",\"endTimeNanos\":\"1610269800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":38}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610269800000000000\",\"endTimeNanos\":\"1610269860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610269860000000000\",\"endTimeNanos\":\"1610269921000000000\",\"value\":[{\"mapVal\":[],\"intVal\":31}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610269921000000000\",\"endTimeNanos\":\"1610269980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610273400000000000\",\"endTimeNanos\":\"1610273460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610273460000000000\",\"endTimeNanos\":\"1610273520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610273820000000000\",\"endTimeNanos\":\"1610273880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610274420000000000\",\"endTimeNanos\":\"1610274480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610274480000000000\",\"endTimeNanos\":\"1610274540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610274720000000000\",\"endTimeNanos\":\"1610274780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610277000000000000\",\"endTimeNanos\":\"1610277060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610277060000000000\",\"endTimeNanos\":\"1610277120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610277120000000000\",\"endTimeNanos\":\"1610277180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610277660000000000\",\"endTimeNanos\":\"1610277720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610278020000000000\",\"endTimeNanos\":\"1610278080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610278200000000000\",\"endTimeNanos\":\"1610278260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610278680000000000\",\"endTimeNanos\":\"1610278740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610278740000000000\",\"endTimeNanos\":\"1610278800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":32}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610278980000000000\",\"endTimeNanos\":\"1610279040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610279520000000000\",\"endTimeNanos\":\"1610279580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610279940000000000\",\"endTimeNanos\":\"1610280000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280060000000000\",\"endTimeNanos\":\"1610280120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280120000000000\",\"endTimeNanos\":\"1610280180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280240000000000\",\"endTimeNanos\":\"1610280300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280480000000000\",\"endTimeNanos\":\"1610280540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280600000000000\",\"endTimeNanos\":\"1610280660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280720000000000\",\"endTimeNanos\":\"1610280780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280780000000000\",\"endTimeNanos\":\"1610280840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280840000000000\",\"endTimeNanos\":\"1610280900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610280900000000000\",\"endTimeNanos\":\"1610280960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":36}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281080000000000\",\"endTimeNanos\":\"1610281140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281200000000000\",\"endTimeNanos\":\"1610281260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281500000000000\",\"endTimeNanos\":\"1610281560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281620000000000\",\"endTimeNanos\":\"1610281680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":61}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281680000000000\",\"endTimeNanos\":\"1610281740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281800000000000\",\"endTimeNanos\":\"1610281860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":64}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610281980000000000\",\"endTimeNanos\":\"1610282040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610282040000000000\",\"endTimeNanos\":\"1610282100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":53}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610282760000000000\",\"endTimeNanos\":\"1610282820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610282820000000000\",\"endTimeNanos\":\"1610282880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610283180000000000\",\"endTimeNanos\":\"1610283240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610284200000000000\",\"endTimeNanos\":\"1610284260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610284260000000000\",\"endTimeNanos\":\"1610284320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":35}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610284860000000000\",\"endTimeNanos\":\"1610284920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610285760000000000\",\"endTimeNanos\":\"1610285820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610286180000000000\",\"endTimeNanos\":\"1610286240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610286960000000000\",\"endTimeNanos\":\"1610287020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610292000000000000\",\"endTimeNanos\":\"1610292060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":33}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610292060000000000\",\"endTimeNanos\":\"1610292120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610294940000000000\",\"endTimeNanos\":\"1610295000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":42}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610295300000000000\",\"endTimeNanos\":\"1610295360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610295480000000000\",\"endTimeNanos\":\"1610295540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610295540000000000\",\"endTimeNanos\":\"1610295600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610295600000000000\",\"endTimeNanos\":\"1610295660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610299080000000000\",\"endTimeNanos\":\"1610299140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610300580000000000\",\"endTimeNanos\":\"1610300640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":45}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610300880000000000\",\"endTimeNanos\":\"1610300940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610300940000000000\",\"endTimeNanos\":\"1610301000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610301240000000000\",\"endTimeNanos\":\"1610301300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610301300000000000\",\"endTimeNanos\":\"1610301360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":36}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610302020000000000\",\"endTimeNanos\":\"1610302080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610302260000000000\",\"endTimeNanos\":\"1610302320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610302920000000000\",\"endTimeNanos\":\"1610302980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610305740000000000\",\"endTimeNanos\":\"1610305800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610305800000000000\",\"endTimeNanos\":\"1610305860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":39}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610305860000000000\",\"endTimeNanos\":\"1610305920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610306340000000000\",\"endTimeNanos\":\"1610306400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610306400000000000\",\"endTimeNanos\":\"1610306460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610306460000000000\",\"endTimeNanos\":\"1610306520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610306520000000000\",\"endTimeNanos\":\"1610306580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":16}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610308860000000000\",\"endTimeNanos\":\"1610308920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610309640000000000\",\"endTimeNanos\":\"1610309700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610310240000000000\",\"endTimeNanos\":\"1610310300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610310600000000000\",\"endTimeNanos\":\"1610310660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610313142744\",\"startTimeNanos\":\"1610311680000000000\",\"endTimeNanos\":\"1610311740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610312940000000000\",\"endTimeNanos\":\"1610313000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610313660000000000\",\"endTimeNanos\":\"1610313720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610314020000000000\",\"endTimeNanos\":\"1610314080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610314560000000000\",\"endTimeNanos\":\"1610314620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610314620000000000\",\"endTimeNanos\":\"1610314680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610316000000000000\",\"endTimeNanos\":\"1610316060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":16}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610317200000000000\",\"endTimeNanos\":\"1610317320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610317320000000000\",\"endTimeNanos\":\"1610318820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610318820000000000\",\"endTimeNanos\":\"1610320260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610320260000000000\",\"endTimeNanos\":\"1610321040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610321040000000000\",\"endTimeNanos\":\"1610321400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610321400000000000\",\"endTimeNanos\":\"1610324880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610324880000000000\",\"endTimeNanos\":\"1610326260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610326260000000000\",\"endTimeNanos\":\"1610327040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610327040000000000\",\"endTimeNanos\":\"1610327280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610327280000000000\",\"endTimeNanos\":\"1610328600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610328600000000000\",\"endTimeNanos\":\"1610328720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610328720000000000\",\"endTimeNanos\":\"1610329620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610329620000000000\",\"endTimeNanos\":\"1610331240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610331240000000000\",\"endTimeNanos\":\"1610332080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610332080000000000\",\"endTimeNanos\":\"1610332800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610332800000000000\",\"endTimeNanos\":\"1610334360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610334360000000000\",\"endTimeNanos\":\"1610334540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610334540000000000\",\"endTimeNanos\":\"1610335320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610335320000000000\",\"endTimeNanos\":\"1610337120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610337120000000000\",\"endTimeNanos\":\"1610337900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610337900000000000\",\"endTimeNanos\":\"1610338920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610338920000000000\",\"endTimeNanos\":\"1610343060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610343060000000000\",\"endTimeNanos\":\"1610343960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610343960000000000\",\"endTimeNanos\":\"1610344320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610344320000000000\",\"endTimeNanos\":\"1610345520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610345520000000000\",\"endTimeNanos\":\"1610346000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610346000000000000\",\"endTimeNanos\":\"1610347200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610347200000000000\",\"endTimeNanos\":\"1610347500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610347620000000000\",\"endTimeNanos\":\"1610347680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610347680000000000\",\"endTimeNanos\":\"1610347740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610348280000000000\",\"endTimeNanos\":\"1610348340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610348340000000000\",\"endTimeNanos\":\"1610348400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610348400000000000\",\"endTimeNanos\":\"1610348460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":52}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610348940000000000\",\"endTimeNanos\":\"1610349000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":15}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610349720000000000\",\"endTimeNanos\":\"1610349780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610350140000000000\",\"endTimeNanos\":\"1610350200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610350860000000000\",\"endTimeNanos\":\"1610350920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610350920000000000\",\"endTimeNanos\":\"1610350980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610351460000000000\",\"endTimeNanos\":\"1610351520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610351520000000000\",\"endTimeNanos\":\"1610351580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610351880000000000\",\"endTimeNanos\":\"1610351940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610352180000000000\",\"endTimeNanos\":\"1610352240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610353080000000000\",\"endTimeNanos\":\"1610353140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610353860000000000\",\"endTimeNanos\":\"1610353920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610354460000000000\",\"endTimeNanos\":\"1610354520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610354880000000000\",\"endTimeNanos\":\"1610354940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610355360000000000\",\"endTimeNanos\":\"1610355420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610355720000000000\",\"endTimeNanos\":\"1610355780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610355780000000000\",\"endTimeNanos\":\"1610355840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":3}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610356080000000000\",\"endTimeNanos\":\"1610356140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610356140000000000\",\"endTimeNanos\":\"1610356200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":33}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610356200000000000\",\"endTimeNanos\":\"1610356260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":53}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610356260000000000\",\"endTimeNanos\":\"1610356320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":49}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610356620000000000\",\"endTimeNanos\":\"1610356680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610357460000000000\",\"endTimeNanos\":\"1610357520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610357940000000000\",\"endTimeNanos\":\"1610358000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610358600000000000\",\"endTimeNanos\":\"1610358660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610358660000000000\",\"endTimeNanos\":\"1610358720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610358720000000000\",\"endTimeNanos\":\"1610358780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610359200000000000\",\"endTimeNanos\":\"1610359260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610359260000000000\",\"endTimeNanos\":\"1610359320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610360700000000000\",\"endTimeNanos\":\"1610360760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610360880000000000\",\"endTimeNanos\":\"1610360940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610361120000000000\",\"endTimeNanos\":\"1610361180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":27}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610361660000000000\",\"endTimeNanos\":\"1610361720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610362080000000000\",\"endTimeNanos\":\"1610362140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":5}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610362260000000000\",\"endTimeNanos\":\"1610362321000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610362740000000000\",\"endTimeNanos\":\"1610362800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610363220000000000\",\"endTimeNanos\":\"1610363280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":29}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610363280000000000\",\"endTimeNanos\":\"1610363340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610363400000000000\",\"endTimeNanos\":\"1610363460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610363640000000000\",\"endTimeNanos\":\"1610363700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610364000000000000\",\"endTimeNanos\":\"1610364060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":39}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610364060000000000\",\"endTimeNanos\":\"1610364120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":3}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365140000000000\",\"endTimeNanos\":\"1610365200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":34}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365200000000000\",\"endTimeNanos\":\"1610365260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":37}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365260000000000\",\"endTimeNanos\":\"1610365320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365380000000000\",\"endTimeNanos\":\"1610365440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365440000000000\",\"endTimeNanos\":\"1610365500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365500000000000\",\"endTimeNanos\":\"1610365560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610365560000000000\",\"endTimeNanos\":\"1610365620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610367240000000000\",\"endTimeNanos\":\"1610367300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610367480000000000\",\"endTimeNanos\":\"1610367540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610367600000000000\",\"endTimeNanos\":\"1610367660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610368740000000000\",\"endTimeNanos\":\"1610368800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":28}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610369340000000000\",\"endTimeNanos\":\"1610369400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610370840000000000\",\"endTimeNanos\":\"1610370900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610371200000000000\",\"endTimeNanos\":\"1610371260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610372520000000000\",\"endTimeNanos\":\"1610372580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610375340000000000\",\"endTimeNanos\":\"1610375400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610375520000000000\",\"endTimeNanos\":\"1610375580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610375580000000000\",\"endTimeNanos\":\"1610375640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610375820000000000\",\"endTimeNanos\":\"1610375880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610376720000000000\",\"endTimeNanos\":\"1610376780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610377800000000000\",\"endTimeNanos\":\"1610377860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610377860000000000\",\"endTimeNanos\":\"1610377920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":23}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610378460000000000\",\"endTimeNanos\":\"1610378520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610378520000000000\",\"endTimeNanos\":\"1610378580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610378700000000000\",\"endTimeNanos\":\"1610378760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610378760000000000\",\"endTimeNanos\":\"1610378820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":29}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610379000000000000\",\"endTimeNanos\":\"1610379060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610379660000000000\",\"endTimeNanos\":\"1610379720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":27}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610380140000000000\",\"endTimeNanos\":\"1610380200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610380620000000000\",\"endTimeNanos\":\"1610380680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610382540000000000\",\"endTimeNanos\":\"1610382600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610383440000000000\",\"endTimeNanos\":\"1610383500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610384580000000000\",\"endTimeNanos\":\"1610384640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387100000000000\",\"endTimeNanos\":\"1610387160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387220000000000\",\"endTimeNanos\":\"1610387280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387520000000000\",\"endTimeNanos\":\"1610387580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387700000000000\",\"endTimeNanos\":\"1610387760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387760000000000\",\"endTimeNanos\":\"1610387820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610387880000000000\",\"endTimeNanos\":\"1610387940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610390460000000000\",\"endTimeNanos\":\"1610390520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610391540000000000\",\"endTimeNanos\":\"1610391600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610391600000000000\",\"endTimeNanos\":\"1610391660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610392200000000000\",\"endTimeNanos\":\"1610392260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610392260000000000\",\"endTimeNanos\":\"1610392320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610392620000000000\",\"endTimeNanos\":\"1610392680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610393160000000000\",\"endTimeNanos\":\"1610393220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610393820000000000\",\"endTimeNanos\":\"1610393880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610394000000000000\",\"endTimeNanos\":\"1610394060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610394060000000000\",\"endTimeNanos\":\"1610394120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":23}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610394420000000000\",\"endTimeNanos\":\"1610394480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610394480000000000\",\"endTimeNanos\":\"1610394540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610395380000000000\",\"endTimeNanos\":\"1610395440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610403300000000000\",\"endTimeNanos\":\"1610403360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610404140000000000\",\"endTimeNanos\":\"1610404200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610404260000000000\",\"endTimeNanos\":\"1610404320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610405040000000000\",\"endTimeNanos\":\"1610405100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610406120000000000\",\"endTimeNanos\":\"1610406180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610406240000000000\",\"endTimeNanos\":\"1610406300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610407200000000000\",\"endTimeNanos\":\"1610407260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610408640000000000\",\"endTimeNanos\":\"1610408760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610408760000000000\",\"endTimeNanos\":\"1610409780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610409780000000000\",\"endTimeNanos\":\"1610411280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610411280000000000\",\"endTimeNanos\":\"1610413980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610413980000000000\",\"endTimeNanos\":\"1610414580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610414580000000000\",\"endTimeNanos\":\"1610415300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610415300000000000\",\"endTimeNanos\":\"1610416800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610416800000000000\",\"endTimeNanos\":\"1610419440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610419440000000000\",\"endTimeNanos\":\"1610419620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610419620000000000\",\"endTimeNanos\":\"1610422260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610422260000000000\",\"endTimeNanos\":\"1610425980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610425980000000000\",\"endTimeNanos\":\"1610427960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610427960000000000\",\"endTimeNanos\":\"1610431740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610431740000000000\",\"endTimeNanos\":\"1610432700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610432700000000000\",\"endTimeNanos\":\"1610433420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610437112571\",\"startTimeNanos\":\"1610433420000000000\",\"endTimeNanos\":\"1610433660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610433780000000000\",\"endTimeNanos\":\"1610433840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":52}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610433840000000000\",\"endTimeNanos\":\"1610433900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610434380000000000\",\"endTimeNanos\":\"1610434440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":44}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610434740000000000\",\"endTimeNanos\":\"1610434800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610434800000000000\",\"endTimeNanos\":\"1610434860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":44}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610434860000000000\",\"endTimeNanos\":\"1610434920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610436060000000000\",\"endTimeNanos\":\"1610436120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610436420000000000\",\"endTimeNanos\":\"1610436480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610437920000000000\",\"endTimeNanos\":\"1610437980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610437980000000000\",\"endTimeNanos\":\"1610438040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":55}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610438040000000000\",\"endTimeNanos\":\"1610438100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610438340000000000\",\"endTimeNanos\":\"1610438400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610438400000000000\",\"endTimeNanos\":\"1610438460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":61}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610439120000000000\",\"endTimeNanos\":\"1610439180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610439180000000000\",\"endTimeNanos\":\"1610439240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":35}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610439360000000000\",\"endTimeNanos\":\"1610439420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":64}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610439420000000000\",\"endTimeNanos\":\"1610439480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":31}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610439480000000000\",\"endTimeNanos\":\"1610439540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610445463996\",\"startTimeNanos\":\"1610440020000000000\",\"endTimeNanos\":\"1610440080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610441580000000000\",\"endTimeNanos\":\"1610441640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610444700000000000\",\"endTimeNanos\":\"1610444760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":28}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610447520000000000\",\"endTimeNanos\":\"1610447580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610447640000000000\",\"endTimeNanos\":\"1610447700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610448720000000000\",\"endTimeNanos\":\"1610448780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610451360000000000\",\"endTimeNanos\":\"1610451420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610451420000000000\",\"endTimeNanos\":\"1610451480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610451480000000000\",\"endTimeNanos\":\"1610451540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":39}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610452380000000000\",\"endTimeNanos\":\"1610452440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610452560000000000\",\"endTimeNanos\":\"1610452620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":24}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610452620000000000\",\"endTimeNanos\":\"1610452680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610456216698\",\"startTimeNanos\":\"1610453520000000000\",\"endTimeNanos\":\"1610453580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456100000000000\",\"endTimeNanos\":\"1610456160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456160000000000\",\"endTimeNanos\":\"1610456220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":77}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456220000000000\",\"endTimeNanos\":\"1610456280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456280000000000\",\"endTimeNanos\":\"1610456340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456340000000000\",\"endTimeNanos\":\"1610456400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":35}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456400000000000\",\"endTimeNanos\":\"1610456460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":40}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456460000000000\",\"endTimeNanos\":\"1610456520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":33}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456520000000000\",\"endTimeNanos\":\"1610456580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":118}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456580000000000\",\"endTimeNanos\":\"1610456640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":123}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456640000000000\",\"endTimeNanos\":\"1610456700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":119}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456700000000000\",\"endTimeNanos\":\"1610456760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":105}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456820000000000\",\"endTimeNanos\":\"1610456880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":24}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456880000000000\",\"endTimeNanos\":\"1610456940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":82}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610459941346\",\"startTimeNanos\":\"1610456940000000000\",\"endTimeNanos\":\"1610457000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":116}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457000000000000\",\"endTimeNanos\":\"1610457060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":109}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457060000000000\",\"endTimeNanos\":\"1610457120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457120000000000\",\"endTimeNanos\":\"1610457180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":102}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457180000000000\",\"endTimeNanos\":\"1610457240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":91}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457240000000000\",\"endTimeNanos\":\"1610457300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":109}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457300000000000\",\"endTimeNanos\":\"1610457360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457480000000000\",\"endTimeNanos\":\"1610457540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457540000000000\",\"endTimeNanos\":\"1610457600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":62}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457600000000000\",\"endTimeNanos\":\"1610457660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":87}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457660000000000\",\"endTimeNanos\":\"1610457720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":70}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457720000000000\",\"endTimeNanos\":\"1610457780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":88}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457780000000000\",\"endTimeNanos\":\"1610457840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":98}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457840000000000\",\"endTimeNanos\":\"1610457900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":102}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457900000000000\",\"endTimeNanos\":\"1610457960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":93}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610457960000000000\",\"endTimeNanos\":\"1610458020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":113}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610458020000000000\",\"endTimeNanos\":\"1610458080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":91}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463530392\",\"startTimeNanos\":\"1610458080000000000\",\"endTimeNanos\":\"1610458140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":108}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458140000000000\",\"endTimeNanos\":\"1610458200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":110}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458200000000000\",\"endTimeNanos\":\"1610458260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":41}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458260000000000\",\"endTimeNanos\":\"1610458320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":45}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458320000000000\",\"endTimeNanos\":\"1610458380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":102}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458380000000000\",\"endTimeNanos\":\"1610458440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":80}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458440000000000\",\"endTimeNanos\":\"1610458500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":108}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458500000000000\",\"endTimeNanos\":\"1610458560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":110}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458560000000000\",\"endTimeNanos\":\"1610458620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":74}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458620000000000\",\"endTimeNanos\":\"1610458680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":98}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458680000000000\",\"endTimeNanos\":\"1610458740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":101}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458740000000000\",\"endTimeNanos\":\"1610458800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":97}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458800000000000\",\"endTimeNanos\":\"1610458860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":99}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458860000000000\",\"endTimeNanos\":\"1610458920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":70}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458920000000000\",\"endTimeNanos\":\"1610458980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":80}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610458980000000000\",\"endTimeNanos\":\"1610459040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":113}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610459040000000000\",\"endTimeNanos\":\"1610459100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":110}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610459100000000000\",\"endTimeNanos\":\"1610459160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":107}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463533636\",\"startTimeNanos\":\"1610459160000000000\",\"endTimeNanos\":\"1610459220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":114}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459220000000000\",\"endTimeNanos\":\"1610459280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":91}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459280000000000\",\"endTimeNanos\":\"1610459340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":104}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459340000000000\",\"endTimeNanos\":\"1610459400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":108}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459400000000000\",\"endTimeNanos\":\"1610459460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459460000000000\",\"endTimeNanos\":\"1610459520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":105}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459520000000000\",\"endTimeNanos\":\"1610459580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":112}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459580000000000\",\"endTimeNanos\":\"1610459640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":111}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459640000000000\",\"endTimeNanos\":\"1610459700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":110}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459700000000000\",\"endTimeNanos\":\"1610459760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":116}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459760000000000\",\"endTimeNanos\":\"1610459820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":113}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459820000000000\",\"endTimeNanos\":\"1610459880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":84}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610463535626\",\"startTimeNanos\":\"1610459880000000000\",\"endTimeNanos\":\"1610459940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":66}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"derived:com.google.step_count.delta:com.withings.wiscale2:Walking-steps\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610459940000000000\",\"endTimeNanos\":\"1610460000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610460000000000000\",\"endTimeNanos\":\"1610460060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610460060000000000\",\"endTimeNanos\":\"1610460120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610460240000000000\",\"endTimeNanos\":\"1610460300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610461620000000000\",\"endTimeNanos\":\"1610461680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610462280000000000\",\"endTimeNanos\":\"1610462340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610462760000000000\",\"endTimeNanos\":\"1610462820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":5}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610464980000000000\",\"endTimeNanos\":\"1610465040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610466120000000000\",\"endTimeNanos\":\"1610466180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610466420000000000\",\"endTimeNanos\":\"1610466480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610466480000000000\",\"endTimeNanos\":\"1610466540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":38}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467260000000000\",\"endTimeNanos\":\"1610467320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467320000000000\",\"endTimeNanos\":\"1610467380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467380000000000\",\"endTimeNanos\":\"1610467440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":83}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467440000000000\",\"endTimeNanos\":\"1610467500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":55}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467560000000000\",\"endTimeNanos\":\"1610467620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":90}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467620000000000\",\"endTimeNanos\":\"1610467680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":27}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610467680000000000\",\"endTimeNanos\":\"1610467740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":33}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610472060000000000\",\"endTimeNanos\":\"1610472120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610473020000000000\",\"endTimeNanos\":\"1610473080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610473320000000000\",\"endTimeNanos\":\"1610473380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610473620000000000\",\"endTimeNanos\":\"1610473680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":32}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610473680000000000\",\"endTimeNanos\":\"1610473740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":80}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610473740000000000\",\"endTimeNanos\":\"1610473800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":28}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610474520000000000\",\"endTimeNanos\":\"1610474580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610474640000000000\",\"endTimeNanos\":\"1610474700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610474700000000000\",\"endTimeNanos\":\"1610474760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610475480000000000\",\"endTimeNanos\":\"1610475540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610478216209\",\"startTimeNanos\":\"1610476680000000000\",\"endTimeNanos\":\"1610476740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610477880000000000\",\"endTimeNanos\":\"1610477940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478000000000000\",\"endTimeNanos\":\"1610478060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478180000000000\",\"endTimeNanos\":\"1610478240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":56}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478240000000000\",\"endTimeNanos\":\"1610478300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478300000000000\",\"endTimeNanos\":\"1610478360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478840000000000\",\"endTimeNanos\":\"1610478900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478900000000000\",\"endTimeNanos\":\"1610478960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":48}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610478960000000000\",\"endTimeNanos\":\"1610479020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610479140000000000\",\"endTimeNanos\":\"1610479200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610479380000000000\",\"endTimeNanos\":\"1610479440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610479440000000000\",\"endTimeNanos\":\"1610479500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610479500000000000\",\"endTimeNanos\":\"1610479560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610480760000000000\",\"endTimeNanos\":\"1610480820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610482500000000000\",\"endTimeNanos\":\"1610482560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610489017290\",\"startTimeNanos\":\"1610484540000000000\",\"endTimeNanos\":\"1610484600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610487600000000000\",\"endTimeNanos\":\"1610487660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610487840000000000\",\"endTimeNanos\":\"1610487900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610488500000000000\",\"endTimeNanos\":\"1610488560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610488560000000000\",\"endTimeNanos\":\"1610488620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610488620000000000\",\"endTimeNanos\":\"1610488680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610489280000000000\",\"endTimeNanos\":\"1610489340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610489880000000000\",\"endTimeNanos\":\"1610489940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610490000000000000\",\"endTimeNanos\":\"1610490060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610490360000000000\",\"endTimeNanos\":\"1610490420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610490420000000000\",\"endTimeNanos\":\"1610490480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610491260000000000\",\"endTimeNanos\":\"1610491380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610491380000000000\",\"endTimeNanos\":\"1610493300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610493300000000000\",\"endTimeNanos\":\"1610494800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610494800000000000\",\"endTimeNanos\":\"1610495580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610495580000000000\",\"endTimeNanos\":\"1610496120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610496120000000000\",\"endTimeNanos\":\"1610497140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610497140000000000\",\"endTimeNanos\":\"1610497500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610497500000000000\",\"endTimeNanos\":\"1610498220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610498220000000000\",\"endTimeNanos\":\"1610502420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610502420000000000\",\"endTimeNanos\":\"1610502720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610502720000000000\",\"endTimeNanos\":\"1610504700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610504700000000000\",\"endTimeNanos\":\"1610505420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610505420000000000\",\"endTimeNanos\":\"1610505660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610505660000000000\",\"endTimeNanos\":\"1610506020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610506020000000000\",\"endTimeNanos\":\"1610508960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610508960000000000\",\"endTimeNanos\":\"1610510040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610510040000000000\",\"endTimeNanos\":\"1610510820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610510820000000000\",\"endTimeNanos\":\"1610512260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610512260000000000\",\"endTimeNanos\":\"1610512980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610512980000000000\",\"endTimeNanos\":\"1610514540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610514540000000000\",\"endTimeNanos\":\"1610515080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610515080000000000\",\"endTimeNanos\":\"1610515680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610515680000000000\",\"endTimeNanos\":\"1610517720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610517720000000000\",\"endTimeNanos\":\"1610518680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610522725398\",\"startTimeNanos\":\"1610518680000000000\",\"endTimeNanos\":\"1610520120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610520180000000000\",\"endTimeNanos\":\"1610520240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":47}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610520960000000000\",\"endTimeNanos\":\"1610521020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":35}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610521020000000000\",\"endTimeNanos\":\"1610521080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":23}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610522220000000000\",\"endTimeNanos\":\"1610522280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":34}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610522460000000000\",\"endTimeNanos\":\"1610522520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610523300000000000\",\"endTimeNanos\":\"1610523360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610524080000000000\",\"endTimeNanos\":\"1610524140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610524980000000000\",\"endTimeNanos\":\"1610525040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610525400000000000\",\"endTimeNanos\":\"1610525460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610525520000000000\",\"endTimeNanos\":\"1610525580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610525580000000000\",\"endTimeNanos\":\"1610525640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610525640000000000\",\"endTimeNanos\":\"1610525700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610526960000000000\",\"endTimeNanos\":\"1610527020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610527200000000000\",\"endTimeNanos\":\"1610527260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528100000000000\",\"endTimeNanos\":\"1610528160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528160000000000\",\"endTimeNanos\":\"1610528220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528220000000000\",\"endTimeNanos\":\"1610528280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":15}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528280000000000\",\"endTimeNanos\":\"1610528340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528340000000000\",\"endTimeNanos\":\"1610528400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610528400000000000\",\"endTimeNanos\":\"1610528460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610529420000000000\",\"endTimeNanos\":\"1610529481000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610530860000000000\",\"endTimeNanos\":\"1610530920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610530980000000000\",\"endTimeNanos\":\"1610531040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":15}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610531220000000000\",\"endTimeNanos\":\"1610531280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610531700000000000\",\"endTimeNanos\":\"1610531760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610531760000000000\",\"endTimeNanos\":\"1610531820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610531940000000000\",\"endTimeNanos\":\"1610532000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610532960000000000\",\"endTimeNanos\":\"1610533020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610533080000000000\",\"endTimeNanos\":\"1610533140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610533140000000000\",\"endTimeNanos\":\"1610533200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610533440000000000\",\"endTimeNanos\":\"1610533500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610533740000000000\",\"endTimeNanos\":\"1610533800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610534280000000000\",\"endTimeNanos\":\"1610534340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610534340000000000\",\"endTimeNanos\":\"1610534400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610534700000000000\",\"endTimeNanos\":\"1610534760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610534940000000000\",\"endTimeNanos\":\"1610535000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610535120000000000\",\"endTimeNanos\":\"1610535180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610535480000000000\",\"endTimeNanos\":\"1610535540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610535900000000000\",\"endTimeNanos\":\"1610535960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":47}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610535960000000000\",\"endTimeNanos\":\"1610536020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610536020000000000\",\"endTimeNanos\":\"1610536080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":26}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610536080000000000\",\"endTimeNanos\":\"1610536141000000000\",\"value\":[{\"mapVal\":[],\"intVal\":26}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610537940000000000\",\"endTimeNanos\":\"1610538000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610538000000000000\",\"endTimeNanos\":\"1610538060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610538060000000000\",\"endTimeNanos\":\"1610538120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":65}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610538360000000000\",\"endTimeNanos\":\"1610538420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610538900000000000\",\"endTimeNanos\":\"1610538960000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610539080000000000\",\"endTimeNanos\":\"1610539140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":39}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610539140000000000\",\"endTimeNanos\":\"1610539200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":27}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610539200000000000\",\"endTimeNanos\":\"1610539260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610539260000000000\",\"endTimeNanos\":\"1610539320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610540400000000000\",\"endTimeNanos\":\"1610540460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":26}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610540460000000000\",\"endTimeNanos\":\"1610540520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610540820000000000\",\"endTimeNanos\":\"1610540880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610541840000000000\",\"endTimeNanos\":\"1610541900000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610542200000000000\",\"endTimeNanos\":\"1610542260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610542260000000000\",\"endTimeNanos\":\"1610542320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":29}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610542320000000000\",\"endTimeNanos\":\"1610542380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610542680000000000\",\"endTimeNanos\":\"1610542740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610543520000000000\",\"endTimeNanos\":\"1610543580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610543640000000000\",\"endTimeNanos\":\"1610543700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610545140000000000\",\"endTimeNanos\":\"1610545200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610545440000000000\",\"endTimeNanos\":\"1610545500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610545560000000000\",\"endTimeNanos\":\"1610545620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610545740000000000\",\"endTimeNanos\":\"1610545800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610546160000000000\",\"endTimeNanos\":\"1610546220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610547120000000000\",\"endTimeNanos\":\"1610547180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610547300000000000\",\"endTimeNanos\":\"1610547360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610548440000000000\",\"endTimeNanos\":\"1610548500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610549160000000000\",\"endTimeNanos\":\"1610549220000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610549700000000000\",\"endTimeNanos\":\"1610549760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610550120000000000\",\"endTimeNanos\":\"1610550180000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610552040000000000\",\"endTimeNanos\":\"1610552100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610552280000000000\",\"endTimeNanos\":\"1610552340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610552340000000000\",\"endTimeNanos\":\"1610552400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610553300000000000\",\"endTimeNanos\":\"1610553360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610553360000000000\",\"endTimeNanos\":\"1610553420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610553780000000000\",\"endTimeNanos\":\"1610553840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":30}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610554440000000000\",\"endTimeNanos\":\"1610554500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610556300000000000\",\"endTimeNanos\":\"1610556360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610556360000000000\",\"endTimeNanos\":\"1610556420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":53}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610556420000000000\",\"endTimeNanos\":\"1610556480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":32}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610556600000000000\",\"endTimeNanos\":\"1610556660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610556720000000000\",\"endTimeNanos\":\"1610556780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610557860000000000\",\"endTimeNanos\":\"1610557920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610558460000000000\",\"endTimeNanos\":\"1610558520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610560680000000000\",\"endTimeNanos\":\"1610560740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610560740000000000\",\"endTimeNanos\":\"1610560800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610563020000000000\",\"endTimeNanos\":\"1610563080000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610563440000000000\",\"endTimeNanos\":\"1610563500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610564520000000000\",\"endTimeNanos\":\"1610564580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":25}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610564640000000000\",\"endTimeNanos\":\"1610564700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610565360000000000\",\"endTimeNanos\":\"1610565420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":41}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610565720000000000\",\"endTimeNanos\":\"1610565780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610566260000000000\",\"endTimeNanos\":\"1610566320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":38}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610566320000000000\",\"endTimeNanos\":\"1610566380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":1}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610566740000000000\",\"endTimeNanos\":\"1610566800000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610567100000000000\",\"endTimeNanos\":\"1610567160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610567640000000000\",\"endTimeNanos\":\"1610567701000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610567701000000000\",\"endTimeNanos\":\"1610567760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610568300000000000\",\"endTimeNanos\":\"1610568360000000000\",\"value\":[{\"mapVal\":[],\"intVal\":20}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610569440000000000\",\"endTimeNanos\":\"1610569500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610569560000000000\",\"endTimeNanos\":\"1610569620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610570760000000000\",\"endTimeNanos\":\"1610570820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":24}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610572020000000000\",\"endTimeNanos\":\"1610572140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610572140000000000\",\"endTimeNanos\":\"1610573820000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610573820000000000\",\"endTimeNanos\":\"1610574540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610574540000000000\",\"endTimeNanos\":\"1610576400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610576400000000000\",\"endTimeNanos\":\"1610576580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610576580000000000\",\"endTimeNanos\":\"1610578320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610578320000000000\",\"endTimeNanos\":\"1610579520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610579520000000000\",\"endTimeNanos\":\"1610580600000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610580600000000000\",\"endTimeNanos\":\"1610581140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610581140000000000\",\"endTimeNanos\":\"1610581860000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610581860000000000\",\"endTimeNanos\":\"1610583420000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610583420000000000\",\"endTimeNanos\":\"1610583660000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610583660000000000\",\"endTimeNanos\":\"1610585760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610585760000000000\",\"endTimeNanos\":\"1610587020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610587020000000000\",\"endTimeNanos\":\"1610587200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610587200000000000\",\"endTimeNanos\":\"1610587740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610587740000000000\",\"endTimeNanos\":\"1610587920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610587920000000000\",\"endTimeNanos\":\"1610589540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610589540000000000\",\"endTimeNanos\":\"1610591280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610591280000000000\",\"endTimeNanos\":\"1610592000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610592000000000000\",\"endTimeNanos\":\"1610592480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610592480000000000\",\"endTimeNanos\":\"1610593560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610593560000000000\",\"endTimeNanos\":\"1610594400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610594400000000000\",\"endTimeNanos\":\"1610596440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610596440000000000\",\"endTimeNanos\":\"1610597280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610597280000000000\",\"endTimeNanos\":\"1610598240000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610598240000000000\",\"endTimeNanos\":\"1610599680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610599680000000000\",\"endTimeNanos\":\"1610601300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610601300000000000\",\"endTimeNanos\":\"1610602200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610602200000000000\",\"endTimeNanos\":\"1610602260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610602260000000000\",\"endTimeNanos\":\"1610603400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610603400000000000\",\"endTimeNanos\":\"1610604120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610604120000000000\",\"endTimeNanos\":\"1610605740000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610605740000000000\",\"endTimeNanos\":\"1610606700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610606700000000000\",\"endTimeNanos\":\"1610608020000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610610604318\",\"startTimeNanos\":\"1610608020000000000\",\"endTimeNanos\":\"1610608200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610608260000000000\",\"endTimeNanos\":\"1610608320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":41}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610608320000000000\",\"endTimeNanos\":\"1610608380000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610608860000000000\",\"endTimeNanos\":\"1610608920000000000\",\"value\":[{\"mapVal\":[],\"intVal\":29}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610608920000000000\",\"endTimeNanos\":\"1610608980000000000\",\"value\":[{\"mapVal\":[],\"intVal\":54}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610608980000000000\",\"endTimeNanos\":\"1610609040000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610609580000000000\",\"endTimeNanos\":\"1610609641000000000\",\"value\":[{\"mapVal\":[],\"intVal\":11}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610609641000000000\",\"endTimeNanos\":\"1610609700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":21}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610612520000000000\",\"endTimeNanos\":\"1610612580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610613060000000000\",\"endTimeNanos\":\"1610613120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610613240000000000\",\"endTimeNanos\":\"1610613300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610615040000000000\",\"endTimeNanos\":\"1610615100000000000\",\"value\":[{\"mapVal\":[],\"intVal\":22}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610615100000000000\",\"endTimeNanos\":\"1610615160000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610615340000000000\",\"endTimeNanos\":\"1610615400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610615460000000000\",\"endTimeNanos\":\"1610615520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":17}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610615640000000000\",\"endTimeNanos\":\"1610615700000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610618340000000000\",\"endTimeNanos\":\"1610618400000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610618400000000000\",\"endTimeNanos\":\"1610618460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610618460000000000\",\"endTimeNanos\":\"1610618520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":18}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610621820000000000\",\"endTimeNanos\":\"1610621880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":23}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610622720000000000\",\"endTimeNanos\":\"1610622780000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610623140000000000\",\"endTimeNanos\":\"1610623200000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610623200000000000\",\"endTimeNanos\":\"1610623260000000000\",\"value\":[{\"mapVal\":[],\"intVal\":53}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610623260000000000\",\"endTimeNanos\":\"1610623320000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610623380000000000\",\"endTimeNanos\":\"1610623440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":9}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624220000000000\",\"endTimeNanos\":\"1610624280000000000\",\"value\":[{\"mapVal\":[],\"intVal\":12}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624400000000000\",\"endTimeNanos\":\"1610624460000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624460000000000\",\"endTimeNanos\":\"1610624520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624520000000000\",\"endTimeNanos\":\"1610624580000000000\",\"value\":[{\"mapVal\":[],\"intVal\":14}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624580000000000\",\"endTimeNanos\":\"1610624640000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610624700000000000\",\"endTimeNanos\":\"1610624760000000000\",\"value\":[{\"mapVal\":[],\"intVal\":19}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610625060000000000\",\"endTimeNanos\":\"1610625120000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610626560000000000\",\"endTimeNanos\":\"1610626620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610626620000000000\",\"endTimeNanos\":\"1610626680000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610627280000000000\",\"endTimeNanos\":\"1610627340000000000\",\"value\":[{\"mapVal\":[],\"intVal\":16}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610628420000000000\",\"endTimeNanos\":\"1610628480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":7}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610629080000000000\",\"endTimeNanos\":\"1610629140000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610629380000000000\",\"endTimeNanos\":\"1610629440000000000\",\"value\":[{\"mapVal\":[],\"intVal\":16}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610629440000000000\",\"endTimeNanos\":\"1610629500000000000\",\"value\":[{\"mapVal\":[],\"intVal\":1}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610629500000000000\",\"endTimeNanos\":\"1610629560000000000\",\"value\":[{\"mapVal\":[],\"intVal\":13}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610632025362\",\"startTimeNanos\":\"1610629560000000000\",\"endTimeNanos\":\"1610629620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":37}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610636453496\",\"startTimeNanos\":\"1610631780000000000\",\"endTimeNanos\":\"1610631840000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610636453496\",\"startTimeNanos\":\"1610633820000000000\",\"endTimeNanos\":\"1610633880000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610636453496\",\"startTimeNanos\":\"1610633880000000000\",\"endTimeNanos\":\"1610633940000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610636453496\",\"startTimeNanos\":\"1610633940000000000\",\"endTimeNanos\":\"1610634000000000000\",\"value\":[{\"mapVal\":[],\"intVal\":3}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610634480000000000\",\"endTimeNanos\":\"1610634540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":6}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610636460000000000\",\"endTimeNanos\":\"1610636520000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610637480000000000\",\"endTimeNanos\":\"1610637540000000000\",\"value\":[{\"mapVal\":[],\"intVal\":19}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610637660000000000\",\"endTimeNanos\":\"1610637720000000000\",\"value\":[{\"mapVal\":[],\"intVal\":26}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610638560000000000\",\"endTimeNanos\":\"1610638620000000000\",\"value\":[{\"mapVal\":[],\"intVal\":8}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610640000000000000\",\"endTimeNanos\":\"1610640060000000000\",\"value\":[{\"mapVal\":[],\"intVal\":0}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610640240000000000\",\"endTimeNanos\":\"1610640300000000000\",\"value\":[{\"mapVal\":[],\"intVal\":10}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"},{\"modifiedTimeMillis\":\"1610654530301\",\"startTimeNanos\":\"1610640420000000000\",\"endTimeNanos\":\"1610640480000000000\",\"value\":[{\"mapVal\":[],\"intVal\":35}],\"dataTypeName\":\"com.google.step_count.delta\",\"originDataSourceId\":\"raw:com.google.step_count.delta:com.withings.wiscale2:com.google.step_count.delta-Withings\"}]}";

        let points = serde_json::from_str::<Points>(STEPS_JSON).unwrap();
        let mut steps = GoogleFitSyncProvider::points_to_steps(points);
        steps.sort_by(|a, b| a.date.cmp(&b.date));

        assert_eq!(steps.len(), 15);
        assert_eq!(
            steps
                .get(0)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-10T21:12:22Z"), 1201))
        );
        assert_eq!(
            steps
                .get(1)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T07:38:32Z"), 1493))
        );
        assert_eq!(
            steps
                .get(2)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T09:57:43Z"), 515))
        );
        assert_eq!(
            steps
                .get(3)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T12:56:56Z"), 184))
        );
        assert_eq!(
            steps
                .get(4)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T13:59:01Z"), 934))
        );
        assert_eq!(
            steps
                .get(5)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T14:58:50Z"), 1368))
        );
        assert_eq!(
            steps
                .get(6)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T14:58:53Z"), 1659))
        );
        assert_eq!(
            steps
                .get(7)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T14:58:55Z"), 1132))
        );
        assert_eq!(
            steps
                .get(8)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T19:03:36Z"), 640))
        );
        assert_eq!(
            steps
                .get(9)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-12T22:03:37Z"), 225))
        );
        assert_eq!(
            steps
                .get(10)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-13T07:25:25Z"), 77))
        );
        assert_eq!(
            steps
                .get(11)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-14T07:50:04Z"), 1341))
        );
        assert_eq!(
            steps
                .get(12)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-14T13:47:05Z"), 575))
        );
        assert_eq!(
            steps
                .get(13)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-14T15:00:53Z"), 23))
        );
        assert_eq!(
            steps
                .get(14)
                .map(|s| (s.date.format_iso8601().unwrap().to_string(), s.steps)),
            Some((String::from("2021-01-14T20:02:10Z"), 104))
        );
    }

    #[test]
    fn test_weights_decode() {
        static WEIGHTS_JSON: &str = "{\"minStartTimeNs\":\"0\",\"maxEndTimeNs\":\"1610750104462000000\",\"dataSourceId\":\"derived:com.google.weight:com.google.android.gms:merge_weight\",\"point\":[{\"modifiedTimeMillis\":\"1600884655879\",\"startTimeNanos\":\"1600884621152000000\",\"endTimeNanos\":\"1600884621152000000\",\"value\":[{\"mapVal\":[],\"fpVal\":73}],\"dataTypeName\":\"com.google.weight\",\"originDataSourceId\":\"raw:com.google.weight:com.google.android.apps.fitness:user_input\"},{\"modifiedTimeMillis\":\"1600884655879\",\"startTimeNanos\":\"1600884630908000000\",\"endTimeNanos\":\"1600884630908000000\",\"value\":[{\"mapVal\":[],\"fpVal\":69}],\"dataTypeName\":\"com.google.weight\",\"originDataSourceId\":\"raw:com.google.weight:com.google.android.apps.fitness:user_input\"},{\"modifiedTimeMillis\":\"1602833332972\",\"startTimeNanos\":\"1602832513832000000\",\"endTimeNanos\":\"1602832513832000000\",\"value\":[{\"mapVal\":[],\"fpVal\":68.0999984741211}],\"dataTypeName\":\"com.google.weight\",\"originDataSourceId\":\"raw:com.google.weight:com.google.android.apps.fitness:user_input\"},{\"modifiedTimeMillis\":\"1603116262806\",\"startTimeNanos\":\"1603112574003000000\",\"endTimeNanos\":\"1603112574003000000\",\"value\":[{\"mapVal\":[],\"fpVal\":70}],\"dataTypeName\":\"com.google.weight\",\"originDataSourceId\":\"raw:com.google.weight:com.google.android.apps.fitness:user_input\"}]}";

        let points = serde_json::from_str::<Points>(WEIGHTS_JSON).unwrap();
        let mut weights = GoogleFitSyncProvider::points_to_weights(points);
        weights.sort_by(|a, b| a.date.cmp(&b.date));

        assert_eq!(weights.len(), 4);

        assert_eq!(
            weights.get(0).map(|w| (
                w.date.format_iso8601().unwrap().to_string(),
                w.weight.get::<kilogram>()
            )),
            Some((String::from("2020-09-23T18:10:55Z"), 73.0))
        );
        assert_eq!(
            weights.get(1).map(|w| (
                w.date.format_iso8601().unwrap().to_string(),
                w.weight.get::<kilogram>()
            )),
            Some((String::from("2020-09-23T18:10:55Z"), 69.0))
        );
        assert_eq!(
            weights.get(2).map(|w| (
                w.date.format_iso8601().unwrap().to_string(),
                w.weight.get::<kilogram>()
            )),
            Some((String::from("2020-10-16T07:28:52Z"), 68.1))
        );
        assert_eq!(
            weights.get(3).map(|w| (
                w.date.format_iso8601().unwrap().to_string(),
                w.weight.get::<kilogram>()
            )),
            Some((String::from("2020-10-19T14:04:22Z"), 70.0))
        );
    }
}
